// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! # rpgpie 🦀️🔐🥧
//!
//! rpgpie is an experimental higher level [OpenPGP](https://www.rfc-editor.org/rfc/rfc4880)
//! library based on [rPGP](https://github.com/rpgp/rpgp).
//!
//! It exposes a convenient API for some simple common OpenPGP operations.
//!
//! rpgpie implements semantics for OpenPGP expiration and revocation mechanisms, key flags,
//! and applies some amount of policy (to limit reliance on obsolete algorithms).
//!
//! Main goals of rpgpie include simplicity and collaboration 🥳.
//!
//! ## "OpenPGP for application developers"
//!
//! rpgpie aims to apply the terminology outlined in the
//! ["OpenPGP for application developers"](https://openpgp.dev/) documentation.
//!
//! ## Stateless OpenPGP (SOP)
//!
//! See [rsop](https://crates.io/crates/rsop) for a
//! [stateless OpenPGP (SOP)](https://datatracker.ietf.org/doc/draft-dkg-openpgp-stateless-cli/)
//! tool based on rpgpie.

pub mod certificate;
pub(crate) mod key;
pub mod message;
pub mod policy;
pub mod signature;
pub mod tsk;
pub(crate) mod util;

pub use crate::key::ComponentKeyPub;

/// rpgpie version, via Cargo.toml
pub const VERSION: &str = env!("CARGO_PKG_VERSION");

/// version of the rpgp dependency we built against
pub const RPGP_VERSION: &str = pgp::VERSION;

/// Error type for this crate
#[derive(thiserror::Error, Debug)]
#[non_exhaustive]
pub enum Error {
    #[error("rPGP error: {0}")]
    Rpgp(pgp::errors::Error),

    #[error("IO error: {0}")]
    Io(std::io::Error),

    #[error("Internal error: {0}")]
    Message(String),

    #[error("No binding signature for the primary key")]
    NoPrimaryBinding,
}

impl From<pgp::errors::Error> for Error {
    fn from(e: pgp::errors::Error) -> Self {
        match e {
            pgp::errors::Error::IOError(io) => Error::Io(io),
            _ => Error::Rpgp(e),
        }
    }
}

impl From<std::io::Error> for Error {
    fn from(value: std::io::Error) -> Self {
        Error::Io(value)
    }
}

impl From<pgp::SubkeyParamsBuilderError> for Error {
    fn from(value: pgp::SubkeyParamsBuilderError) -> Self {
        Error::Rpgp(pgp::errors::Error::Message(format!(
            "SubkeyParamsBuilderError: {}",
            value
        )))
    }
}

impl From<pgp::SecretKeyParamsBuilderError> for Error {
    fn from(value: pgp::SecretKeyParamsBuilderError) -> Self {
        Error::Rpgp(pgp::errors::Error::Message(format!(
            "SecretKeyParamsBuilderError: {}",
            value
        )))
    }
}
