// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/user_prefs/tracked/tracked_preference_helper.h"

#include "base/logging.h"
#include "base/metrics/histogram.h"
#include "base/metrics/histogram_macros.h"
#include "components/user_prefs/tracked/tracked_preference_histogram_names.h"

TrackedPreferenceHelper::TrackedPreferenceHelper(
    const std::string& pref_path,
    size_t reporting_id,
    size_t reporting_ids_count,
    PrefHashFilter::EnforcementLevel enforcement_level,
    PrefHashFilter::ValueType value_type)
    : pref_path_(pref_path),
      reporting_id_(reporting_id),
      reporting_ids_count_(reporting_ids_count),
      enforce_(enforcement_level == PrefHashFilter::ENFORCE_ON_LOAD),
      personal_(value_type == PrefHashFilter::VALUE_PERSONAL) {
}

TrackedPreferenceHelper::ResetAction TrackedPreferenceHelper::GetAction(
    PrefHashStoreTransaction::ValueState value_state) const {
  switch (value_state) {
    case PrefHashStoreTransaction::UNCHANGED:
      // Desired case, nothing to do.
      return DONT_RESET;
    case PrefHashStoreTransaction::CLEARED:
      // Unfortunate case, but there is nothing we can do.
      return DONT_RESET;
    case PrefHashStoreTransaction::TRUSTED_NULL_VALUE:  // Falls through.
    case PrefHashStoreTransaction::TRUSTED_UNKNOWN_VALUE:
      // It is okay to seed the hash in this case.
      return DONT_RESET;
    case PrefHashStoreTransaction::SECURE_LEGACY:
      // Accept secure legacy device ID based hashes.
      return DONT_RESET;
    case PrefHashStoreTransaction::UNSUPPORTED:
      NOTREACHED()
          << "GetAction should not be called with an UNSUPPORTED value state";
      return DONT_RESET;
    case PrefHashStoreTransaction::UNTRUSTED_UNKNOWN_VALUE:  // Falls through.
    case PrefHashStoreTransaction::CHANGED:
      return enforce_ ? DO_RESET : WANTED_RESET;
  }
  NOTREACHED() << "Unexpected PrefHashStoreTransaction::ValueState: "
               << value_state;
  return DONT_RESET;
}

bool TrackedPreferenceHelper::IsPersonal() const {
  return personal_;
}

void TrackedPreferenceHelper::ReportValidationResult(
    PrefHashStoreTransaction::ValueState value_state,
    base::StringPiece validation_type_suffix) const {
  const char* histogram_name = nullptr;
  switch (value_state) {
    case PrefHashStoreTransaction::UNCHANGED:
      histogram_name = user_prefs::tracked::kTrackedPrefHistogramUnchanged;
      break;
    case PrefHashStoreTransaction::CLEARED:
      histogram_name = user_prefs::tracked::kTrackedPrefHistogramCleared;
      break;
    case PrefHashStoreTransaction::SECURE_LEGACY:
      histogram_name =
          user_prefs::tracked::kTrackedPrefHistogramMigratedLegacyDeviceId;
      break;
    case PrefHashStoreTransaction::CHANGED:
      histogram_name = user_prefs::tracked::kTrackedPrefHistogramChanged;
      break;
    case PrefHashStoreTransaction::UNTRUSTED_UNKNOWN_VALUE:
      histogram_name = user_prefs::tracked::kTrackedPrefHistogramInitialized;
      break;
    case PrefHashStoreTransaction::TRUSTED_UNKNOWN_VALUE:
      histogram_name =
          user_prefs::tracked::kTrackedPrefHistogramTrustedInitialized;
      break;
    case PrefHashStoreTransaction::TRUSTED_NULL_VALUE:
      histogram_name =
          user_prefs::tracked::kTrackedPrefHistogramNullInitialized;
      break;
    case PrefHashStoreTransaction::UNSUPPORTED:
      NOTREACHED() << "ReportValidationResult should not be called with an "
                      "UNSUPPORTED value state";
      return;
  }
  DCHECK(histogram_name);

  std::string full_histogram_name(histogram_name);
  if (!validation_type_suffix.empty()) {
    full_histogram_name.push_back('.');
    validation_type_suffix.AppendToString(&full_histogram_name);
  }

  // Using FactoryGet to allow dynamic histogram names. This is equivalent to
  // UMA_HISTOGRAM_ENUMERATION(name, reporting_id_, reporting_ids_count_);
  base::HistogramBase* histogram = base::LinearHistogram::FactoryGet(
      full_histogram_name, 1, reporting_ids_count_, reporting_ids_count_ + 1,
      base::HistogramBase::kUmaTargetedHistogramFlag);
  histogram->Add(reporting_id_);
}

void TrackedPreferenceHelper::ReportAction(ResetAction reset_action) const {
  switch (reset_action) {
    case DONT_RESET:
      // No report for DONT_RESET.
      break;
    case WANTED_RESET:
      UMA_HISTOGRAM_ENUMERATION(
          user_prefs::tracked::kTrackedPrefHistogramWantedReset, reporting_id_,
          reporting_ids_count_);
      break;
    case DO_RESET:
      UMA_HISTOGRAM_ENUMERATION(user_prefs::tracked::kTrackedPrefHistogramReset,
                                reporting_id_, reporting_ids_count_);
      break;
  }
}

void TrackedPreferenceHelper::ReportSplitPreferenceChangedCount(
    size_t count) const {
  // The histogram below is an expansion of the UMA_HISTOGRAM_COUNTS_100 macro
  // adapted to allow for a dynamically suffixed histogram name.
  // Note: The factory creates and owns the histogram.
  base::HistogramBase* histogram = base::LinearHistogram::FactoryGet(
      user_prefs::tracked::kTrackedSplitPrefHistogramChanged + pref_path_, 1,
      100,  // Allow counts up to 100.
      101, base::HistogramBase::kUmaTargetedHistogramFlag);
  histogram->Add(count);
}
